<?php

namespace Wisebee;

/**
 * Class API. Handles API requests to Wisebee Platform
 *
 * @author Uldis Zvirbulis <uldis.zvirbulis@mtcmedia.co.uk>
 * @package Wisebee
 */
class API
{
    private $apiKey;
    private $endpoint;
    private $method;
    private $headers;
    private $params = [];
    private $response;
    private $result;

    const METHOD_GET = 'GET';
    const METHOD_POST = 'POST';
    const METHOD_PATCH = 'PATCH';
    const METHOD_DELETE = 'DELETE';

    /**
     * Sets API key and default headers
     *
     * API constructor.
     * @param string $endpoint
     * @param string $apiKey
     */
    public function __construct(string $endpoint, string $apiKey)
    {
        $this->endpoint = $endpoint;
        $this->apiKey = $apiKey;
        $this->headers = [
            'Authorization: Bearer ' . $this->apiKey,
            'Content-Type: application/json',
        ];
    }

    /**
     * API GET request
     *
     * @param string $uri
     * @throws WisebeeException
     */
    public function get(string $uri)
    {
        $this->setMethod(self::METHOD_GET);
        $this->call($uri);
    }

    /**
     * API POST request
     *
     * @param string $uri
     * @throws WisebeeException
     */
    public function post(string $uri)
    {
        $this->setMethod(self::METHOD_POST);
        $this->call($uri);
    }

    /**
     * API PATCH request
     *
     * @param string $uri
     * @throws WisebeeException
     */
    public function patch(string $uri)
    {
        $this->setMethod(self::METHOD_PATCH);
        $this->call($uri);
    }

    /**
     * API DELETE request
     *
     * @param string $uri
     * @throws WisebeeException
     */
    public function delete(string $uri)
    {
        $this->setMethod(self::METHOD_DELETE);
        $this->call($uri);
    }

    /**
     * Performs API request
     *
     * @param string $uri
     * @throws WisebeeException
     */
    public function call(string $uri)
    {
        if (empty($uri)) {
            throw new WisebeeException('API URI missing');
        }

        if (empty($this->headers)) {
            throw new WisebeeException('API Headers missing');
        }

        if (empty($this->apiKey)) {
            throw new WisebeeException('API Key missing');
        }

        $curl = curl_init();
        if ($this->method === self::METHOD_GET) {
            $uri .= '?' . http_build_query($this->params);
        } else {
            curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($this->params));
        }

        if ($this->method !== self::METHOD_POST) {
            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $this->method);
        }

        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_URL, $this->endpoint . $uri);
        curl_setopt($curl, CURLOPT_HTTPHEADER, $this->headers);
        curl_setopt($curl, CURLOPT_POST, true);

        $this->response = curl_exec($curl);
        $this->result = curl_getinfo($curl);

        curl_close($curl);
    }

    /**
     * Returns HTTP response code of the request
     *
     * @return false|int
     */
    public function getResultCode()
    {
        if (empty($this->result)) {
            return false;
        }
        return (int)$this->result['http_code'];
    }

    /**
     * Returns request response output
     *
     * @return mixed
     */
    public function getResponse()
    {
        return $this->response;
    }

    /**
     * Sets request headers. Merges them with existing ones
     *
     * @param array $headers
     * @throws WisebeeException
     */
    public function setHeaders(array $headers)
    {
        if (!is_array($headers)) {
            throw new WisebeeException('Headers must be an array');
        }
        foreach ($headers as $key => $value) {
            $this->headers[] = $key . ': ' . $value;
        }
    }

    /**
     * Sets POST fields
     *
     * @param array $params
     */
    public function setParams(array $params)
    {
        $this->params = $params;
    }

    /**
     * Sets Method
     *
     * @param string $method
     */
    private function setMethod(string $method)
    {
        $this->method = $method;
    }
}