<?php

namespace Wisebee;

use Wisebee\Resources\Resource;

/**
 * Class Wisebee. Helper class which serves as a layer between API and Website
 *
 * @author Uldis Zvirbulis <uldis.zvirbulis@mtcmedia.co.uk>
 * @package Wisebee
 */
class Wisebee
{
    /** @var API $api */
    private $api;

    /** @var string $error */
    private $error = '';

    /** @var int $count */
    private $count = 0;

    /**
     * Wisebee class constructor
     *
     * @param string $endpoint
     * @param string $apiKey
     */
    public function __construct(string $endpoint, string $apiKey)
    {
        $this->api = new API($endpoint, $apiKey);
    }

    /**
     * Gets resource
     *
     * @param $resourceClass
     * @param $id
     * @return bool|Resource
     */
    public function get($resourceClass, $id)
    {
        /** @var Resource $resource */
        $resource = new $resourceClass($this);

        try {
            $this->api->get($resource->uriBase . '/' . $id);
        } catch (WisebeeException $e) {
            $this->error = $e->getMessage();
            return false;
        }
        if (!$data = $this->processRequest()) {
            return false;
        }
        return (new $resourceClass())->build($data);
    }

    /**
     * Gets a list of resource
     *
     * @param $resourceClass
     * @param $params
     * @return bool|array
     */
    public function list($resourceClass, $params)
    {
        /** @var Resource $resource */
        $resource = new $resourceClass($this);

        try {
            $this->api->setParams($params);
            $this->api->get($resource->uriBase);
        } catch (WisebeeException $e) {
            $this->error = $e->getMessage();
            return false;
        }
        if (!$data = $this->processRequest()) {
            return false;
        }
        $this->count = $data->count;
        foreach ($data->list as $key => $object) {
            $data->list[$key] = (new $resourceClass())->build($object);
        }
        return $data->list;
    }

    /**
     * Creates a resource
     *
     * @param $resourceClass
     * @param $data
     * @return bool|object
     */
    public function create($resourceClass, $data)
    {
        /** @var Resource $resource */
        $resource = new $resourceClass($this);

        try {
            $this->api->setParams($data);
            $this->api->post($resource->uriBase);
        } catch (WisebeeException $e) {
            $this->error = $e->getMessage();
            return false;
        }
        if (!$data = $this->processRequest()) {
            return false;
        }
        return (new $resourceClass())->build($data);
    }

    /**
     * Updates a resource
     *
     * @param $resourceClass
     * @param $id
     * @param $data
     * @return bool|object
     */
    public function update($resourceClass, $id, $data)
    {
        /** @var Resource $resource */
        $resource = new $resourceClass($this);

        try {
            $this->api->setParams($data);
            $this->api->patch($resource->uriBase . '/' . $id);
        } catch (WisebeeException $e) {
            $this->error = $e->getMessage();
            return false;
        }
        if (!$data = $this->processRequest()) {
            return false;
        }
        return (new $resourceClass())->build($data);
    }

    /**
     * Deletes a resource
     *
     * @param $resourceClass
     * @param $id
     * @return bool|object
     */
    public function delete($resourceClass, $id)
    {
        /** @var Resource $resource */
        $resource = new $resourceClass($this);

        try {
            $this->api->delete($resource->uriBase . '/' . $id);
        } catch (WisebeeException $e) {
            $this->error = $e->getMessage();
            return false;
        }
        return $this->processRequest();
    }

    /**
     * A custom API call for non-standard REST requests
     *
     * @param $method
     * @param $uri
     * @param array $params
     * @return bool|object
     */
    public function customApiCall($method, $uri, array $params = [])
    {
        try {
            if (!empty($params)) {
                $this->api->setParams($params);
            }
            $this->api->$method($uri);
        } catch (WisebeeException $e) {
            $this->error = $e->getMessage();
            return false;
        }
        return $this->processRequest();
    }

    /**
     * Gets error
     *
     * @return string
     */
    public function getError(): string
    {
        return $this->error;
    }

    /**
     * Gets number of records
     *
     * @return int
     */
    public function getCount(): int
    {
        return $this->count;
    }

    /**
     * Processes request. Returns either data array on success or false on failed request
     * Sets errors on failed requests
     *
     * @return object|bool
     */
    private function processRequest()
    {
        $this->error = '';
        $resultCode = $this->api->getResultCode();
        $response = json_decode($this->api->getResponse());

        if (!empty($response->error)) {
            $this->error = $response->error;
            return false;
        }
        if ($resultCode === 200) {
            return $response->data;
        }
        $this->error = 'Request failed with HTTP code: ' . $resultCode . ' and response: ' .
            print_r($response, true);
        return false;
    }
}