<?php

namespace Wisebee;

use Closure;

/**
 * Class Wisebee. Helper class which serves as a layer between API and Website
 *
 * @author Uldis Zvirbulis <uldis.zvirbulis@mtcmedia.co.uk>
 * @package Wisebee
 */
class Wisebee
{
    /** @var API $api */
    private $api;

    /** @var array $post */
    private $post = [];

    /**
     * Wisebee class constructor
     *
     * @param string $endpoint
     * @param string $username
     * @param string $apiKey
     */
    public function __construct(string $endpoint, string $username, string $apiKey)
    {
        $this->api = new API($endpoint, $username, $apiKey);
    }

    /**
     * Sets Post parameters
     *
     * @param array $post
     */
    public function setPost(array $post)
    {
        $this->post = $post;
    }

    /**
     * @param Closure $callback
     * @return bool
     * @throws WisebeeException
     */
    public function book(Closure $callback): bool
    {
        // We only go further if the action is 'book'
        if ($this->post['action'] !== 'book') {
            return false;
        }

        if (!$this->validateBooking()) {
            return false;
        }

        $callback($this->post['consultations']);
        return true;
    }

    /**
     * Makes a call to Wisebee server to confirm the request really came from there
     * Just uses the token and public key of the first consultations
     * to find if it exists in Wisebee database
     *
     * @return bool
     * @throws WisebeeException
     */
    public function validateBooking(): bool
    {
        $this->api->setPostFields([
            'token' => $this->post['consultations'][0]['token'],
            'public_key' => $this->post['consultations'][0]['public_key'],
        ]);
        $this->api->call('booking/validate');

        return $this->api->isSuccess();
    }

    /**
     * @param $data
     * @return bool
     * @throws WisebeeException
     */
    public function submitPayment($data): bool
    {
        $this->api->setPostFields($data);
        $this->api->call('booking/payment');
        return $this->api->isSuccess();
    }

    /**
     * @param $token
     * @param $consultationID
     * @return bool
     * @throws WisebeeException
     */
    public function cancelTemporaryConsultation($token, $consultationID): bool
    {
        $this->api->setPostFields([
            'token' => $token,
            'consultation_id' => $consultationID,
        ]);
        $this->api->call('booking/cancel-temporary');
        return $this->api->isSuccess();
    }
}