<?php

namespace Wisebee;

/**
 * Class API. Handles API requests to Wisebee Platform
 *
 * @author Uldis Zvirbulis <uldis.zvirbulis@mtcmedia.co.uk>
 * @package Wisebee
 */
class API
{
    private $username;
    private $apiKey;
    private $endpoint = 'https://www.uldis.consult.mtcdevserver4.com/api/';
    private $headers;
    private $postFields = [];
    private $response;
    private $result;

    /**
     * Sets API username and API key and default headers
     *
     * API constructor.
     * @param string $endpoint
     * @param string $username
     * @param string $apiKey
     */
    public function __construct(string $endpoint, string $username, string $apiKey)
    {
        $this->endpoint = $endpoint;
        $this->username = $username;
        $this->apiKey = $apiKey;
        $this->headers = [
            'Authorization: Bearer ' . $this->apiKey,
            'Content-Type: application/json',
        ];
    }

    /**
     * @param $uri
     * @throws WisebeeException
     */
    public function call($uri)
    {
        if (empty($uri)) {
            throw new WisebeeException('API URI missing');
        }

        if (empty($this->headers)) {
            throw new WisebeeException('API Headers missing');
        }

        if (empty($this->username)) {
            throw new WisebeeException('API Username missing');
        }

        if (empty($this->apiKey)) {
            throw new WisebeeException('API Key missing');
        }

        // Add username for extra bit of security
        $this->postFields['username'] = $this->username;

        $curl = curl_init();
        curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($this->postFields));
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_URL, $this->endpoint . $uri);
        curl_setopt($curl, CURLOPT_HTTPHEADER, $this->headers);
        curl_setopt($curl, CURLOPT_POST, true);

        $this->response = curl_exec($curl);
        $this->result = curl_getinfo($curl);

        curl_close($curl);
    }

    /**
     * Returns HTTP response code of the request
     *
     * @return false|int
     */
    public function getResultCode()
    {
        if (empty($this->result)) {
            return false;
        }
        return (int)$this->result['http_code'];
    }

    /**
     * Returns request response output
     *
     * @return mixed
     */
    public function getResponse()
    {
        return $this->response;
    }

    /**
     * Whether the request was successful
     *
     * @return bool
     */
    public function isSuccess(): bool
    {
        return $this->getResultCode() === 200 && $this->getResponse() === 'OK';
    }

    /**
     * @param $headers
     * @throws WisebeeException
     */
    public function setHeaders($headers)
    {
        if (!is_array($headers)) {
            throw new WisebeeException('Headers must be an array');
        }
        $this->headers = array_merge($this->headers, $headers);
    }

    /**
     * @param $postFields
     * @throws WisebeeException
     */
    public function setPostFields($postFields)
    {
        if (!is_array($postFields)) {
            throw new WisebeeException('Post fields must be an array');
        }
        $this->postFields = $postFields;
    }

    /**
     * Sets Endpoint
     *
     * @param $endpoint
     */
    public function setEndpoint($endpoint)
    {
        $this->endpoint = $endpoint;
    }
}