<?php

namespace OCP;

/**
 * Class API. Handles API requests to Online Consultations Platform
 *
 * @author Uldis Zvirbulis <uldis.zvirbulis@mtcmedia.co.uk>
 * @package OCP
 */
class API
{
    private $username;
    private $apiKey;
    private $endpoint = 'https://www.consult.mtcdevserver4.com/api/';
    private $headers;
    private $postFields = [];
    private $response;
    private $result;

    /**
     * Sets API username and API key and default headers
     *
     * API constructor.
     * @param $username
     * @param $apiKey
     */
    public function __construct($username, $apiKey)
    {
        $this->username = $username;
        $this->apiKey = $apiKey;
        $this->headers = [
            'Authorization: Bearer ' . $this->apiKey,
            'Content-Type: application/json',
        ];
    }

    /**
     * @param $uri
     * @throws OCPException
     */
    public function call($uri)
    {
        if (empty($uri)) {
            throw new OCPException('API URI missing');
        }

        if (empty($this->headers)) {
            throw new OCPException('API Headers missing');
        }

        if (empty($this->username)) {
            throw new OCPException('API Username missing');
        }

        if (empty($this->apiKey)) {
            throw new OCPException('API Key missing');
        }

        // Add username for extra bit of security
        $this->postFields['username'] = $this->username;

        $curl = curl_init();
        curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($this->postFields));
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_URL, $this->endpoint . $uri);
        curl_setopt($curl, CURLOPT_HTTPHEADER, $this->headers);
        curl_setopt($curl, CURLOPT_POST, true);

        $this->response = curl_exec($curl);
        $this->result = curl_getinfo($curl);

        curl_close($curl);
    }

    /**
     * Returns HTTP response code of the request
     *
     * @return false|int
     */
    public function getResultCode()
    {
        if (empty($this->result)) {
            return false;
        }
        return (int)$this->result['http_code'];
    }

    /**
     * Returns request response output
     *
     * @return mixed
     */
    public function getResponse()
    {
        return $this->response;
    }

    /**
     * @param $headers
     * @throws OCPException
     */
    public function setHeaders ($headers)
    {
        if (!is_array($headers)) {
            throw new OCPException('Headers must be an array');
        }
        $this->headers = array_merge($this->headers, $headers);
    }

    /**
     * @param $postFields
     * @throws OCPException
     */
    public function setPostFields ($postFields)
    {
        if (!is_array($postFields)) {
            throw new OCPException('Post fields must be an array');
        }
        $this->postFields = $postFields;
    }

    /**
     * Sets Endpoint
     *
     * @param $endpoint
     */
    public function setEndpoint($endpoint)
    {
        $this->endpoint = $endpoint;
    }
}